#ifndef _lf_topo_map_h_
#define _lf_topo_map_h_

/*
 * This is a purely topological map of the network in a form
 * suitable for sending over the network
 */

/*
 * The topo map is represented in memory as one "struct lf_topo_map"
 * follow by "num_xbars" "struct lf_topo_xbar"s, etc.  The following
 * macros are for getting to each array
 */
#define LF_TOPO_XBAR_ARRAY(TM)						\
  ((struct lf_topo_xbar *)((char *)(TM) + sizeof(struct lf_topo_map)))
#define LF_TOPO_NIC_ARRAY(TM)						\
  ((struct lf_topo_nic *)((char *)LF_TOPO_XBAR_ARRAY(TM)		\
   + (sizeof(struct lf_topo_xbar) *					\
     ntohl(((struct lf_topo_map *)(TM))->num_xbars_32))))
#define LF_TOPO_LINK_ARRAY(TM)						\
  ((struct lf_topo_link *)((char *)LF_TOPO_NIC_ARRAY(TM) 		\
   + (sizeof(struct lf_topo_nic) * 					\
     ntohl(((struct lf_topo_map *)(TM))->num_nics_32))))
#define LF_TOPO_LINK_STATE_ARRAY(TM)					\
  ((lf_topo_link_state_t *)((char *)LF_TOPO_LINK_ARRAY(TM)		\
   + (sizeof(struct lf_topo_link) * 					\
     ntohl(((struct lf_topo_map *)(TM))->num_links_32))))
#define LF_TOPO_LAG_ID_ARRAY(TM)					\
  ((unsigned char *)((char *)LF_TOPO_LINK_STATE_ARRAY(TM)		\
   + (sizeof(lf_topo_link_state_t) * 					\
     ntohl(((struct lf_topo_map *)(TM))->num_links_32))))
#define LF_TOPO_LAG_ID(TM, I)						\
  (LF_TOPO_LAG_ID_ARRAY(TM) + (I)*LF_LAG_ID_LEN)
#define LF_TOPO_HOSTNAME_ARRAY(TM)					\
  ((char *)LF_TOPO_LAG_ID_ARRAY(TM)					\
   + (LF_LAG_ID_LEN *							\
     ntohl(((struct lf_topo_map *)(TM))->num_lag_ids_32)))

#define LF_TOPO_MAP_SIZE(XBARS, NICS, LINKS, LAG_IDS, HOSTNAME_SIZE)	\
   (sizeof(struct lf_topo_map)						\
   + (sizeof(struct lf_topo_xbar) * (XBARS))				\
   + (sizeof(struct lf_topo_nic) * (NICS))				\
   + (sizeof(struct lf_topo_link) * (LINKS))				\
   + (sizeof(lf_topo_link_state_t) * (LINKS))				\
   + (LF_LAG_ID_LEN * (LAG_IDS))					\
   + (HOSTNAME_SIZE))
#define LF_TOPO_LINK_STATE_SIZE(TM)					\
	(ntohl(((struct lf_topo_map *)(TM))->num_links_32) 		\
	 * sizeof(lf_topo_link_state_t))

/*
 * When sending the topological map of the fabric over the wire, the
 * first thing send will be a header defining the map.  Then, the map elements
 * will be sent in order:  1 struct for each xbar, 1 struct for each NIC,
 * and then 1 struct for each link.
 */
struct lf_topo_map {
  uint32_t map_id_32;
  uint8_t mapper_mac_addr[6];	/* MAC address of mapper generating this map */
  uint16_t mapper_level_16;	/* Mapper level */
  uint32_t num_hosts_32;	/* number of hosts in map */
  uint32_t num_xbars_32;
  uint32_t num_nics_32;
  uint32_t num_links_32;
  uint32_t num_lag_ids_32;
  uint32_t hostname_array_size_32;
};

/*
 * Topo node types
 */
enum lf_topo_node_type {
  LF_TOPO_NONE=0,
  LF_TOPO_NIC=1,
  LF_TOPO_XBAR=2
};

/*
 * wire representation of an xbar
 */
struct lf_topo_xbar {
  uint32_t node_id_32;
  uint32_t xbar_id_32;
  uint16_t num_ports_16;
  uint8_t clos_level_8;
  uint8_t quadrant_disable_8;
};

/* xbar node_id is the xbar index in the fabric
 */
#define LF_TOPO_XBAR_NODE_ID(XBAR) (XBAR)

/*
 * Wire representation of a NIC
 */
struct lf_topo_nic {
  uint32_t node_id_32;
  uint8_t mac_addr[6];
  uint8_t num_ports_8;			/* NIC interfaces */
  uint8_t firmware_type_8;		/* GM, MX, 2Z, etc. */
  uint16_t partition_16;		/* partition membership */
  uint16_t mag_id_16;			/* myrinet link aggregation ID */
  uint16_t flags_16;			/* misc flags */
  uint16_t lag_id_index_16;		/* Index into LAG_ID table */
  uint16_t map_distributor_16;		/* who sends this guy's map? */
  uint32_t subfab_mask_32;		/* host sub-fabric membership */
};

/* A NICs node_id is an amalgam of its host index and slot in host */
#define LF_TOPO_NIC_NODE_ID(HID, SLOT) (((HID)<<8)|(SLOT))
#define LF_TOPO_NIC_HOST_ID(NI) ((NI)>>8)
#define LF_TOPO_NIC_HOST_NIC_ID(NI) ((NI)&0xFF)

/*
 * wire representation of a link between two nodes
 */
struct lf_topo_link {
  uint32_t node_index_32[2];	/* index into NIC or xbar array */
  uint32_t verifier_32;		/* node encoding of NIC/port to verify */
};

/*
 * State of a link - index in this array is same as index in link array.
 * It is a seperate array so that it may be sent and updated seperately from
 * the map to reduce network traffic.  There is one link_state per link.
 */
typedef uint8_t lf_topo_link_state_t;

/* definitions for composing and decomposing node_index for a link */
#define LF_TOPO_NODE_INDEX(INDEX, TYPE, PORT) (((INDEX)<<8)|((TYPE)<<6)|(PORT))
#define LF_TOPO_VERIFIER_INDEX(INDEX, WHICH, PORT) \
  	LF_TOPO_NODE_INDEX(INDEX, (WHICH+1), PORT) /* TYPE == which end */
#define LF_TOPO_XBAR_INDEX(I, P) LF_TOPO_NODE_INDEX(I, LF_TOPO_XBAR, P)
#define LF_TOPO_NIC_INDEX(I, P) LF_TOPO_NODE_INDEX(I, LF_TOPO_NIC, P)

#define LF_TOPO_NI_INDEX(NI) ((NI)>>8)
#define LF_TOPO_NI_TYPE(NI) (((NI) >> 6) & 0x03)
#define LF_TOPO_NI_WHICH_END(NI) (LF_TOPO_NI_TYPE(NI) - 1)  /* for verifier */
#define LF_TOPO_NI_PORT(NI) ((NI)&0x3f)
#define LF_TOPO_NI_XBAR(NI) (LF_TOPO_NI_TYPE(NI) == LF_TOPO_XBAR)

/*
 * Special MAC addr for FMS-generated maps
 */
#define FMS_MAPPER_MAC "FMSMAP"

/*
 * prototypes
 */
struct lf_fabric;
struct lf_topo_map *lf_create_topo_map(struct lf_fabric *fp, int *ret_map_size);

#endif /* _lf_topo_map_h_ */
